package com.kangxin.common.byh.base;

import android.content.Context;
import android.view.View;
import android.view.ViewGroup;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.chad.library.adapter.base.BaseQuickAdapter;
import com.chad.library.adapter.base.BaseViewHolder;
import com.kangxin.common.R;
import com.kangxin.common.base.kt.BaseFragment;
import com.kangxin.common.base.kt.IFlushMoreView;
import com.scwang.smartrefresh.layout.SmartRefreshLayout;

import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * 生命周期
 * init（初始化布局信息，查找view，完成页面加载）
 * onInitComplete（开始绑定数据，设置默认事件监听）
 *
 * @param <T> 页面加载的实体
 * @author qinpengL
 */
public abstract class BaseFlushMoreFragment<T> extends BaseFragment implements IFlushMoreView<T> {

    protected BaseQuickAdapter<T, ? extends BaseViewHolder> mAdapter;

    protected RecyclerView vRecyclerView;

    protected SmartRefreshLayout vSmartRefreshLayout;

    /**
     * 获取adapter
     *
     * @return
     */
    protected abstract BaseQuickAdapter<T, ? extends BaseViewHolder> getBaseQuickAdapter();

    /**
     * 获取布局id
     *
     * @return
     */
    @Override
    protected abstract int getLayoutId();

    /**
     * 初始化
     */
    @Override
    public void init() {
        searchView(rootView);
    }

    //初始化完成后回调
    @Override
    public void onInitComplete() {

        //绑定布局
        vRecyclerView.setLayoutManager(new LinearLayoutManager(mContext));
        mAdapter = getBaseQuickAdapter();
        vRecyclerView.setAdapter(mAdapter);

        //关闭加载更多
        mAdapter.setEnableLoadMore(false);
        //刷新事件
        vSmartRefreshLayout.setOnRefreshListener(refreshlayout -> {
            flushOrLoad(false);
        });

        //加载更多
        mAdapter.setOnLoadMoreListener(() -> {
            flushOrLoad(true);
        }, vRecyclerView);

        flushOrLoad(false);
    }

    /**
     * 刷新或者加载
     *
     * @param isGetMore
     */
    protected abstract void flushOrLoad(boolean isGetMore);

    /**
     * 加载布局
     *
     * @param parentView
     * @return
     */
    private View searchView(View parentView) {
        if (parentView instanceof ViewGroup) {
            int childCount = ((ViewGroup) parentView).getChildCount();
            for (int i = 0; i < childCount; i++) {
                View childView = ((ViewGroup) parentView).getChildAt(i);
                if (childView instanceof RecyclerView) {
                    vRecyclerView = (RecyclerView) childView;
                }
                if (childView instanceof SmartRefreshLayout) {
                    vSmartRefreshLayout = (SmartRefreshLayout) childView;
                }
                searchView(childView);
            }
        } else {
            return parentView;
        }
        return null;
    }


    /**
     * 绑定数据
     *
     * @param pageData
     */
    @Override
    public void bindData(List<T> pageData) {
        if (mAdapter == null) {
            mAdapter = getBaseQuickAdapter();
        } else {

            mAdapter.setNewData(pageData);
        }
        mAdapter.setEnableLoadMore(false);

        if (pageData != null) {
            if (pageData.size() == 10) {
                mAdapter.setEnableLoadMore(true);
            }
        }

//        mAdapter.setNewData(pageData);

//        vRecyclerView.setAdapter(mAdapter);
        vSmartRefreshLayout.finishRefresh();
    }

    @NotNull
    @Override
    public Context injectContext() {
        return mContext;
    }

    /**
     * 退出当前fragment
     */
    public void finishCurrentView() {
        pop();
    }

    /**
     * 添加更多
     *
     * @param pageData
     */
    @Override
    public void addMore(List<T> pageData) {
        if (mAdapter == null) {
            mAdapter = getBaseQuickAdapter();
        }

        if (pageData.size() < 10) {
            mAdapter.setEnableLoadMore(false);
        }

        mAdapter.addData(pageData);
        mAdapter.loadMoreComplete();
    }

    /**
     * 没有更多数据
     */
    @Override
    public void noMore() {
        if (mAdapter == null) {
            mAdapter = getBaseQuickAdapter();
        }
        mAdapter.loadMoreEnd();
    }

    @NotNull
    public View getVToolView() {
        return rootView.findViewById(R.id.vToolBar);
    }

    /**
     * 没有数据
     */
    @Override
    public void noData() {
        mAdapter.setNewData(null);
        vSmartRefreshLayout.finishRefresh();
    }
}
