package com.kangxin.common.http.helper;

import android.content.Context;

import com.kangxin.common.http.interceptor.AddCookieInterceptor;
import com.kangxin.common.http.interceptor.Interceptor;
import com.kangxin.common.http.interceptor.SaveCookieInterceptor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import okhttp3.logging.HttpLoggingInterceptor;


public class OkHttpHelper {

    //默认拦截等级
    public static final int DEFAULT_LEVEL = 0;
    //最后拦截
    public static final int LAST_LEVEL = 50;
    //最早拦截
    public static final int EARLIEST_LEVEL = 0;
    //客户端
    private OkHttpClient mOkhttpClient;
    //默认构建
    private OkHttpClient.Builder mBuilder;
    //拦截器列表
    private ArrayList<Interceptor> interceptors = new ArrayList<Interceptor>();

    public OkHttpClient getOkHttpClient() {

        if (mOkhttpClient == null) {
            mOkhttpClient = createOkClient();
        }
        return mOkhttpClient;
    }


    /**
     * 创建构建器
     */
    private void createBuilder() {
        mBuilder = new OkHttpClient.Builder().connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(20, TimeUnit.SECONDS).writeTimeout(20, TimeUnit.SECONDS);
    }

    /**
     * 创建客户端
     *
     * @return
     */
    private OkHttpClient createOkClient() {
        if (mBuilder == null) {
            createBuilder();
        }
        addDefaultInterceptor();
        return mOkhttpClient = mBuilder.build();

    }

    /**
     * 添加拦截器
     *
     * @param interceptor
     * @return
     */
    public OkHttpClient addInterceptor(okhttp3.Interceptor interceptor) {
        return addInterceptor(interceptor, DEFAULT_LEVEL);
    }


    /**
     * 添加拦截器
     *
     * @param interceptor 拦截器
     * @param level       拦截器等级，等级越高，拦截越早
     * @return
     */
    public OkHttpClient addInterceptor(okhttp3.Interceptor interceptor, int level) {

        //ok拦截器以及拦截等级封装
        Interceptor levelInterceptor = new Interceptor(interceptor, level);
        //存入容器
        interceptors.add(levelInterceptor);
        //容器排序
        Collections.sort(interceptors);

        mBuilder.interceptors().clear();

        for (Interceptor interceptor1 : interceptors) {
            mBuilder.addInterceptor(interceptor1.getInterceptor());
        }

        return mOkhttpClient = mBuilder.build();
    }


    /**
     * 添加默认拦截器
     */
    private void addDefaultInterceptor() {
        //日志拦截器

        //添加header拦截器
        //addInterceptor(new RequestHeaderInterceptor());
        addInterceptor(new HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY), LAST_LEVEL);
    }

    /**
     * 添加cookie拦截器
     *
     * @param context
     */
    public void cookieSupport(Context context) {

        addInterceptor(new SaveCookieInterceptor(context));
        addInterceptor(new AddCookieInterceptor(context));
    }


}
