package com.byh.module.onlineoutser.im.utils;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.media.AudioManager;
import android.media.MediaMetadataRetriever;
import android.media.MediaRecorder;
import android.media.ToneGenerator;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.os.Vibrator;
import android.provider.Settings;
import android.util.Log;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;

import androidx.appcompat.app.AlertDialog;
import androidx.fragment.app.FragmentActivity;

import com.byh.module.onlineoutser.im.utils.permissions.Permission;
import com.byh.module.onlineoutser.im.utils.permissions.RxPermissions2;
import com.byh.module.onlineoutser.im.view.RecordPopupWindow;
import com.kangxin.common.R;
import com.kangxin.common.util.StringsUtils;

import java.io.File;

import io.reactivex.functions.Consumer;


/**
 * Created by poetr on 17/11/3.
 */

public class RecordAudioUtils implements MediaRecorder.OnInfoListener, Handler.Callback {

    private static final String TAG = "RecordAudioUtils";
    private static final int TONE_LENGTH_MS = 200;
    private static final float TONE_RELATIVE_VOLUME = 100.0F;
    private static final int MAX_DURATION = 60;

    private View mAnchor;
    private RecordPopupWindow mPopupWindow;
    private Context mContext;
    private float mDy;

    private ToneGenerator mToneGenerator;
    private Object mToneGeneratorLock = new Object();

    private MediaRecorder mRecorder = null;

    private Handler mHandler;

    private Runnable mUpdateVisualizer = new Runnable() {
        @Override
        public void run() {
            if (mRecorder == null) return;
            // get the current amplitude
            int ratio = mRecorder.getMaxAmplitude() / 600;
            int db = 0;// 分贝
            if (ratio > 1)
                db = (int) (20 * Math.log10(ratio));
            // updateMsg in 40 milliseconds
            mPopupWindow.displayAmplitude(db / 4);
            mHandler.postDelayed(this, 150);
        }
    };

    private File mFile;
    private boolean mTimeOut;
    private AudioRecordListener mListener;
    private HandlerThread mTimerThread;
    private Handler mTimerHandler;
    private long mLastEvent;
    private RxPermissions2 rxPermissions;
    private boolean isRequesting;
    private boolean enable = true;
    private AlertDialog mWarnDialog;


    private RecordAudioUtils(Context context, View anchor) {
        mAnchor = anchor;
        this.mContext = context;
        rxPermissions = new RxPermissions2((FragmentActivity) mContext);
        mPopupWindow = new RecordPopupWindow(View.inflate(mContext,
                R.layout.online_voice_rcd_hint_window2, null),
                WindowManager.LayoutParams.MATCH_PARENT,
                WindowManager.LayoutParams.WRAP_CONTENT);
    }

    public static RecordAudioUtils with(Context context, View v) {
        return new RecordAudioUtils(context, v);
    }

    public synchronized static int getVoiceDuration(String path) {
        MediaMetadataRetriever mmr = new MediaMetadataRetriever();
        mmr.setDataSource(path);
        String durationStr = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DURATION);
        if (durationStr == null) {
            return 0;
        }
        Log.i(TAG, "getVoiceDuration: durationStr:" + durationStr);
        final double v = Double.parseDouble(durationStr);
        if (v < 500) {
            return 0;
        } else {
            return (int) Math.round(v / 1000);
        }
    }

    public void setEnable(boolean enable) {
        this.enable = enable;
    }

    public void show() {
        mAnchor.setOnTouchListener(new View.OnTouchListener() {
            @SuppressLint("CheckResult")
            @Override
            public boolean onTouch(View v, final MotionEvent event) {
                if (!enable) {
                    if (event.getActionMasked() == MotionEvent.ACTION_DOWN) {
//                        ToastUtils.INSTANCE.makeShort(StringsUtils.getString(R.string.onlineoutser_shipinzhongwufafasongyuyinxiaoxi));
                    }
                    return true;
                }
                if (System.currentTimeMillis() - mLastEvent < 300) return true;
                if (!rxPermissions.isGranted(Manifest.permission.RECORD_AUDIO)) {
                    if (isRequesting) return true;
                    isRequesting = true;
                    rxPermissions.requestEach(Manifest.permission.RECORD_AUDIO)
                            .subscribe(new Consumer<Permission>() {
                                @Override
                                public void accept(Permission permission) throws Exception {
                                    isRequesting = false;
                                    if (!permission.granted && !permission.shouldShowRequestPermissionRationale) {
                                        showPermissionAlert();
                                    }
                                }
                            });
                    return true;
                }
                switch (event.getActionMasked()) {
                    case MotionEvent.ACTION_DOWN:
                        MediaPlayUtils.checkStatus();
                        mDy = event.getRawY();
                        mPopupWindow.switchStatus(RecordPopupWindow.PREPARE);
                        readyOperation();
                        initMediaRecorder();
                        mPopupWindow.switchStatus(RecordPopupWindow.RECORDING);
                        mPopupWindow.showAtLocation(mAnchor, Gravity.CENTER, 0, 0);
                        break;
                    case MotionEvent.ACTION_MOVE:
                        if (mTimeOut) return true;
                        float y = event.getRawY();
                        if (mDy - y > UiUtil.INSTANCE.dpToPx(20f)) {
                            mPopupWindow.switchStatus(RecordPopupWindow.CANCEL);
                        } else {
                            mPopupWindow.switchStatus(RecordPopupWindow.RECORDING);
                        }
                        break;
                    case MotionEvent.ACTION_UP:
                    case MotionEvent.ACTION_CANCEL:
                        mLastEvent = System.currentTimeMillis();
                        if (mTimeOut) break;
                        if (mFile == null) break;
                        stopRecord();
                        if (mFile.exists() && mFile.length() > 0) {
                            if (mPopupWindow.getStatus() == RecordPopupWindow.CANCEL) {
                                mFile.delete();
                                mPopupWindow.dismiss();
                                break;
                            } else {
                                int mill = getVoiceDuration(mFile.getAbsolutePath());
                                if (mill == 0) {
                                    mFile.delete();
                                    warning(RecordPopupWindow.TOO_SHORT);
                                    break;
                                }
                            }
                            complete();
                            mPopupWindow.dismiss();
                        } else {
                            mFile.delete();
                            warning(RecordPopupWindow.TOO_SHORT);
                            break;
                        }
                        break;
                }
                return true;
            }
        });
    }

    private void showPermissionAlert() {
        if (mWarnDialog == null)
            mWarnDialog = new AlertDialog.Builder(mContext).setTitle(StringsUtils.getString(R.string.onlineoutser_wenxintishi)).setMessage(StringsUtils.getString(R.string.onlineoutser_luyinquanxianjujue___) +
                    StringsUtils.getString(R.string.onlineoutser_qingzai_shezhi_yingyongchengxu_huayitongyishengban_quanxian_shoudongdakailuyinquanxian_))
                    .setNegativeButton(StringsUtils.getString(R.string.onlineoutser_quxiao), new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.dismiss();
                        }
                    })
                    .setPositiveButton(StringsUtils.getString(R.string.onlineoutser_qudakai), new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            Intent intent = new Intent();
                            intent.setAction(Settings.ACTION_APPLICATION_SETTINGS);
                            if (intent.resolveActivity(mContext.getPackageManager()) != null) {
                                mContext.startActivity(intent);
                            }
                            dialog.dismiss();
                        }
                    }).create();
        if (!mWarnDialog.isShowing()) mWarnDialog.show();
    }

    public boolean isRecording() {
        return mPopupWindow.isShowing();
    }

    public void setAudioRecorderListener(AudioRecordListener listener) {
        this.mListener = listener;
    }

    private void complete() {
        if (mListener != null) {
            if (mFile != null) {
                mListener.onAudioComplete(mFile.getAbsolutePath());
            }
        }
    }

    private void warning(int status) {
        mPopupWindow.switchStatus(status);
        mHandler.postDelayed(new Runnable() {
            @Override
            public void run() {
                mPopupWindow.dismiss();
            }
        }, 1000);
    }

    private synchronized void initMediaRecorder() {
        mFile = new File(LocalFileUtils.getVoiceFileDir(mContext) + File.separator + String.valueOf(System.currentTimeMillis()) + ".amr");
        if (mRecorder == null) {
            mRecorder = new MediaRecorder();
            mHandler = new Handler();
            mTimerThread = new HandlerThread("RecordTimer");
        }
        try {
            mRecorder.setAudioSource(MediaRecorder.AudioSource.MIC);
//            mRecorder.setOutputFormat(MediaRecorder.OutputFormat.AAC_ADTS);
            mRecorder.setOutputFormat(MediaRecorder.OutputFormat.DEFAULT);
            mRecorder.setAudioEncoder(MediaRecorder.AudioEncoder.AAC);
            mRecorder.setOutputFile(mFile.getAbsolutePath());
            mRecorder.setMaxDuration(MAX_DURATION * 1000);
            mRecorder.setOnInfoListener(this);
            mRecorder.prepare();
            mRecorder.start();
        } catch (Exception e) {
            mRecorder = null;
            mPopupWindow.dismiss();
        }
        mTimerThread.start();
        mTimerHandler = new Handler(mTimerThread.getLooper(), RecordAudioUtils.this);
        mTimerHandler.sendMessage(mTimerHandler.obtainMessage());
        mHandler.post(mUpdateVisualizer);
        mTimeOut = false;
        mAnchor.setKeepScreenOn(true);
    }

    private synchronized void stopRecord() {
        if (mRecorder != null) {
            try {
                mRecorder.stop();
            } catch (Exception e) {
            }
            mRecorder.release();
            mRecorder = null;
            mHandler.removeCallbacks(mUpdateVisualizer);
            mTimerThread.quit();
            mTimerThread = null;
            mAnchor.setKeepScreenOn(false);
        }
    }


    private void readyOperation() {
        playTone(ToneGenerator.TONE_PROP_BEEP, TONE_LENGTH_MS);
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                stopTone();
            }
        }, TONE_LENGTH_MS);
        vibrate(50L);
    }

    public void playTone(int tone, int durationMs) {
        synchronized (mToneGeneratorLock) {
            initToneGenerator();
            if (mToneGenerator == null) {
                return;
            }

            // Start the new tone (will stop any playing tone)
            mToneGenerator.startTone(tone, durationMs);
        }
    }

    @SuppressLint("MissingPermission")
    public synchronized void vibrate(long milliseconds) {
        Vibrator mVibrator = (Vibrator) mContext.getSystemService(Context.VIBRATOR_SERVICE);
        if (mVibrator == null) {
            return;
        }
        mVibrator.vibrate(milliseconds);
    }

    public void stopTone() {
        if (mToneGenerator != null)
            mToneGenerator.stopTone();
    }

    private void initToneGenerator() {
        AudioManager mAudioManager = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
        if (mToneGenerator == null) {
            try {
                int streamVolume = mAudioManager.getStreamVolume(AudioManager.STREAM_MUSIC);
                int streamMaxVolume = mAudioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
                int volume = (int) (TONE_RELATIVE_VOLUME * (streamVolume / streamMaxVolume));
                mToneGenerator = new ToneGenerator(AudioManager.STREAM_MUSIC, volume);
            } catch (RuntimeException e) {
                mToneGenerator = null;
            }
        }
    }

    @Override
    public void onInfo(MediaRecorder mr, int what, int extra) {
        if (what == MediaRecorder.MEDIA_RECORDER_INFO_MAX_DURATION_REACHED) {
            mTimeOut = true;
            stopRecord();
            mPopupWindow.dismiss();
            complete();
        }
    }

    @Override
    public boolean handleMessage(Message msg) {
        if (!mFile.exists() || mFile.length() == 0) {
            mTimerHandler.sendMessageDelayed(Message.obtain(), 1000);
            return true;
        }
        int curr = getVoiceDuration(mFile.getAbsolutePath());
        final int rest = MAX_DURATION - curr;
        if (rest <= 10) {
            mHandler.post(new Runnable() {
                @Override
                public void run() {
                    if (mPopupWindow.getStatus() == RecordPopupWindow.RECORDING ||
                            mPopupWindow.getStatus() == RecordPopupWindow.ON_TICK) {
                        mPopupWindow.switchStatus(RecordPopupWindow.ON_TICK);
                        mPopupWindow.updateTick(rest);
                    }
                }
            });
        }
        if (rest > 0) {
            mTimerHandler.sendMessageDelayed(Message.obtain(), 1000);
        }
        return true;
    }

    public interface AudioRecordListener {
        void onAudioComplete(String result);
    }
}
