package com.kangxin.common.util;

import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.LayoutRes;
import androidx.recyclerview.widget.RecyclerView;

import com.chad.library.adapter.base.BaseQuickAdapter;
import com.chad.library.adapter.base.BaseViewHolder;
import com.scwang.smartrefresh.layout.SmartRefreshLayout;
import com.scwang.smartrefresh.layout.api.RefreshLayout;
import com.scwang.smartrefresh.layout.listener.OnRefreshListener;

import java.util.ArrayList;
import java.util.List;

/**
 * 刷新的辅助类
 *
 * @author SWY
 * @date 2018/7/6-下午5:06
 */
public class SmartRefreshHelper<E> {

    //第一页的index
    private static final int FIRST_PAGE = 0;
    private final RefreshLayout refreshLayout;
    private final BaseQuickAdapter<E, BaseViewHolder> adapter;
    //每次期望固定请求的数量
    private int perPageCount = 0;
    private int duration = 300;
    private RefreshCallback callback = null;

    private int currentPage = FIRST_PAGE;

    private boolean refreshEnable = true;
    private boolean loadMoreEnable = true;

    private int emptyId = 0;

    private SmartRefreshHelper(RefreshLayout layout, BaseQuickAdapter<E, BaseViewHolder> adapter) {
        refreshLayout = layout;
        this.adapter = adapter;

        refreshLayout.setEnableOverScrollBounce(true);
        refreshLayout.setEnableAutoLoadmore(false);

    }

    public static void pureScrollMode(RefreshLayout layout) {
        layout.setEnablePureScrollMode(true);
        layout.setEnableOverScrollBounce(true);
        layout.setEnableNestedScroll(true);
        layout.setEnablePureScrollMode(true);
    }

    public static <E> SmartRefreshHelper<E> with(SmartRefreshLayout layout, BaseQuickAdapter<E, BaseViewHolder> adapter) {
        return new SmartRefreshHelper<>(layout, adapter);
    }


    public SmartRefreshHelper<E> setPerPageCount(int perPageCount) {
        this.perPageCount = perPageCount;
        return this;
    }

    public SmartRefreshHelper<E> setRefreshEnable(boolean refreshEnable) {
        this.refreshEnable = refreshEnable;
        return this;
    }

    public SmartRefreshHelper<E> setLoadMoreEnable(boolean loadMoreEnable) {
        this.loadMoreEnable = loadMoreEnable;
        return this;
    }

    public SmartRefreshHelper<E> setEmptyId(@LayoutRes int emptyId) {
        this.emptyId = emptyId;
        return this;
    }

    public SmartRefreshHelper<E> init(RefreshCallback refreshCallback) {

        this.callback = refreshCallback;

        refreshLayout.setEnableRefresh(refreshEnable);
        if (refreshEnable) {
            refreshLayout.setOnRefreshListener(new OnRefreshListener() {
                @Override
                public void onRefresh(RefreshLayout refreshLayout) {
                    requestFirstPage(false);
                }
            });
        }

        refreshLayout.setEnableLoadmore(false);

        RecyclerView v = null;
        for (int i = 0; i < refreshLayout.getLayout().getChildCount(); i++) {
            if (refreshLayout.getLayout().getChildAt(i) instanceof RecyclerView)
                v = (RecyclerView) refreshLayout.getLayout().getChildAt(i);
        }

        adapter.setEnableLoadMore(loadMoreEnable);
        if (loadMoreEnable) {
            adapter.setOnLoadMoreListener(new BaseQuickAdapter.RequestLoadMoreListener() {
                @Override
                public void onLoadMoreRequested() {
                    currentPage++;
                    if (callback != null) {
                        callback.doRequestData(currentPage);
                    }
                }
            }, v);
        }
        return this;
    }

    //回调数据刷新
    public SmartRefreshHelper<E> requestFirstPage(boolean loading) {

        //是否显示中心旋转
        if (loading) {
            showCenterLoading();
        }
        currentPage = FIRST_PAGE;
        if (callback != null) {
            callback.doRequestData(currentPage);
        }
        return this;
    }

    public void onSuccess(List<E> data) {
        synchronized (adapter) {

            hideCenterLoading();

            if (currentPage <= FIRST_PAGE) {

                //下拉
                adapter.setNewData(data);
                if (perPageCount > 0) {
                    if (data != null && data.size() == perPageCount) {
                        adapter.loadMoreComplete();
                    } else if (data != null && data.size() > 0) {
                        adapter.loadMoreEnd();
                    } else if (emptyId != 0)
                        adapter.setEmptyView(emptyId, (ViewGroup) refreshLayout);
                }

            } else {
                //上拉
                if (data == null || data.size() == 0) {
                    //当前页加载无数据index-1;
                    currentPage--;
                    if (perPageCount > 0) {
                        adapter.loadMoreEnd();
                    }
                } else if (data.size() == perPageCount) {
                    adapter.addData(data);
                    adapter.loadMoreComplete();
                } else {
                    adapter.addData(data);
                    adapter.loadMoreEnd();
                }
            }
            refreshLayout.finishRefresh(duration, true);
        }
    }

    public void onFailed() {

        synchronized (adapter) {

            hideCenterLoading();

            //下拉
            if (currentPage <= FIRST_PAGE) {
                adapter.setNewData(new ArrayList<>());
                if (perPageCount > 0) {
                    adapter.loadMoreComplete();
                }
            }
            //上拉
            else {
                currentPage--;
                if (perPageCount > 0) {
                    adapter.loadMoreFail();
                }
            }
            refreshLayout.finishRefresh(duration, false);
        }
    }

    private void showCenterLoading() {
        GlobeLoadingHelper.show(((View) refreshLayout).getContext());
    }

    private void hideCenterLoading() {
        GlobeLoadingHelper.dissmiss();
    }

    public interface RefreshCallback {
        void doRequestData(int page);
    }

}
