package com.kangxin.common.http;


import android.content.Context;

import com.google.gson.Gson;
import com.kangxin.common.R;
import com.kangxin.common.http.api.DownloadService;
import com.kangxin.common.http.api.JsonService;
import com.kangxin.common.http.factory.GsonErrorConverterFactory;
import com.kangxin.common.http.helper.OkHttpHelper;
import com.kangxin.common.http.interceptor.DownloadInterceptor;
import com.kangxin.common.http.listener.DownListener;
import com.kangxin.common.util.SchedulesSwitch;
import com.kangxin.common.util.StringsUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.TimeUnit;

import io.reactivex.Observable;
import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.adapter.rxjava2.RxJava2CallAdapterFactory;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * 网络请求数据封装
 *
 * @author qinpeng
 * <p>
 * create date by 2018/6/19 上午9:36
 */
public class RetrofitUtils {

    //实例对象
    private static RetrofitUtils instance;
    //retrofit对象
    private Retrofit mRetrofit;
    //okHttpHelper
    private OkHttpHelper okHttpHelper;
    //okclientbuilder
    private Retrofit.Builder builder;
    //是否切换线程
    private boolean switchThread = false;
    //rx支持
    private boolean rxSupport = false;
    //是否缓存
    private boolean cache = false;
    //是否cookie
    private boolean cookieSupport = false;
    //baseUrl
    private String baseUrl;

    //单例
    private RetrofitUtils() {
    }

    //获取实例
    public static RetrofitUtils getInstance() {
        if (instance == null) {
            instance = new RetrofitUtils();
        }
        return instance;
    }

    private Retrofit.Builder getBuilder(String baseUrl) {
        this.baseUrl = baseUrl;
        return builder = new Retrofit
                .Builder()
                .baseUrl(baseUrl)
                .addConverterFactory(GsonConverterFactory.create(new Gson()))
                .addConverterFactory(GsonErrorConverterFactory.create(new Gson()))
                .addCallAdapterFactory(RxJava2CallAdapterFactory.create());
    }

    //返回retrofit
    public Retrofit createRetrofit(String baseUrl) {

        if (baseUrl == null || baseUrl.equals("")) {
            throw new RuntimeException(StringsUtils.getString(R.string.common_baseurl_shikongde));
        }
        getBuilder(baseUrl);

        return mRetrofit = builder.client(createOkHttpHelper().getOkHttpClient()).build();
    }

    //创建api对象
    public <T> T createApi(Class<T> clazz) {
        if (mRetrofit == null) {
            createRetrofit(null);
        }
        return mRetrofit.create(clazz);
    }


    /**
     * 缓存支持
     *
     * @param cacheSupport 缓存支持
     * @return
     */
    public RetrofitUtils cacheSupport(boolean cacheSupport) {
        // TODO: 2018/6/26 未实现
        this.cache = cacheSupport;
        return this;
    }


    /**
     * 设置是否支持cookie
     *
     * @param cookieSupport cookie支持
     * @return
     */
    public RetrofitUtils cookieSupport(boolean cookieSupport, Context context) {
        this.cookieSupport = cookieSupport;
        okHttpHelper.cookieSupport(context);
        return this;
    }

    /**
     * 添加rx支持库
     *
     * @param rxSupport
     * @return
     */
    public RetrofitUtils rxSupport(boolean rxSupport) {
        this.rxSupport = rxSupport;
        return this;
    }

    /**
     * 是否自动切换线程
     *
     * @param switchThread
     * @return
     */
    public RetrofitUtils switchThread(boolean switchThread) {
        this.switchThread = switchThread;
        return this;
    }


    /**
     * 创建okhttphelper
     *
     * @return
     */
    private OkHttpHelper createOkHttpHelper() {
        if (okHttpHelper == null)
            okHttpHelper = new OkHttpHelper();
        return okHttpHelper;
    }

    /**
     * 添加拦截器
     *
     * @param interceptor 拦截器
     * @param level       拦截器等级，等级越高，越早被拦截
     */
    public void addInterceptor(Interceptor interceptor, int level) {
        if (okHttpHelper == null) {
            createOkHttpHelper();
        }
        mRetrofit = builder.client(okHttpHelper.addInterceptor(interceptor, level)).build();
    }


    /**
     * 传送json数据
     *
     * @param url      接口地址
     * @param json     json字符串
     * @param callback 回调地址
     */
    public void json(String url, String json, Callback<String> callback) {
        RequestBody jsonBody = RequestBody.create(okhttp3.MediaType.parse("application/json; charset=utf-8"), json);
        JsonService api = createApi(JsonService.class);
        api.sendJson(url, jsonBody).enqueue(callback);
    }


    /**
     * 发送json 返回结果为rx包装类型
     *
     * @param url  接口地址
     * @param json json数据
     * @return
     */
    public <T> Observable<T> json(String url, String json) {
        RequestBody jsonBody = RequestBody.create(okhttp3.MediaType.parse("application/json; charset=utf-8"), json);
        JsonService api = createApi(JsonService.class);
        Observable<T> observable = api.<T>sendJsonRx(url, jsonBody);
        if (rxSupport) {
            if (switchThread) {
                observable = observable.compose(SchedulesSwitch.<T>applySchedulers());
            }
        }
        return observable;
    }


    /**
     * 下载文件
     *
     * @param url 下载路径
     */
    public void downloadFile(String url, final String path, final DownListener downListener) {

        OkHttpClient httpClient = new OkHttpClient.Builder()
                .retryOnConnectionFailure(true)
                .connectTimeout(5000, TimeUnit.SECONDS)
                .addInterceptor(new DownloadInterceptor(downListener))
                .build();

        Retrofit retrofit = new Retrofit.Builder()
                .baseUrl(baseUrl)
                .client(httpClient)
                .build();


        DownloadService downloadService = retrofit.create(DownloadService.class);
        downloadService.downloadFile(url).enqueue(new Callback<ResponseBody>() {
            @Override
            public void onResponse(Call<ResponseBody> call, Response<ResponseBody> response) {
                final InputStream inputStream = response.body().byteStream();


                new Thread(new Runnable() {
                    @Override
                    public void run() {
                        writeFile(inputStream, path, downListener);
                        downListener.downSuccess();
                    }
                }).start();

            }

            @Override
            public void onFailure(Call<ResponseBody> call, Throwable t) {
                downListener.downFail();
            }
        });


    }


    private void writeFile(InputStream inputString, String filePath, DownListener listener) {

        File file = new File(filePath);
        if (file.exists()) {
            file.delete();
        }

        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);

            byte[] b = new byte[1024];

            int len;
            while ((len = inputString.read(b)) != -1) {
                fos.write(b, 0, len);
            }
            inputString.close();
            fos.close();

        } catch (FileNotFoundException e) {
            listener.downFail();
        } catch (IOException e) {
            listener.downFail();
        }

    }


}
