/*
 * Copyright 2011 woozzu
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.byh.module.onlineoutser.view;

import android.content.Context;
import android.database.DataSetObserver;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.RectF;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.MotionEvent;
import android.widget.Adapter;
import android.widget.ListAdapter;
import android.widget.ListView;
import android.widget.SectionIndexer;

import androidx.core.content.ContextCompat;

import com.kangxin.common.R;


public class IndexableListView extends ListView {

    private boolean mIsFastScrollEnabled = false;
    private IndexScroller mScroller = null;


    public IndexableListView(Context context) {
        super(context);
    }

    public IndexableListView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    public IndexableListView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    @Override
    public boolean isFastScrollEnabled() {
        return mIsFastScrollEnabled;
    }

    @Override
    public void setFastScrollEnabled(boolean enabled) {

        mIsFastScrollEnabled = enabled;
        if (mIsFastScrollEnabled) {
            if (mScroller == null)
                mScroller = new IndexScroller(getContext(), this);
        }
    }

    @Override
    public void draw(Canvas canvas) {
        super.draw(canvas);
        // Overlay index bar
        if (mScroller != null)
            mScroller.draw(canvas);
    }

    @Override
    public boolean onTouchEvent(MotionEvent ev) {
        // Intercept ListView's touch event
        if (mScroller != null && mScroller.onTouchEvent(ev)) {
            return true;
        }

        return super.onTouchEvent(ev);
    }

    @Override
    public boolean onInterceptTouchEvent(MotionEvent ev) {
        if (mScroller != null && mScroller.contains(ev.getX(), ev.getY()))
            return true;

        return super.onInterceptTouchEvent(ev);
    }


    @Override
    public void setAdapter(ListAdapter adapter) {
        super.setAdapter(adapter);
        if (mScroller != null)
            mScroller.setAdapter(adapter);
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        super.onLayout(changed, l, t, r, b);
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        super.onSizeChanged(w, h, oldw, oldh);
        if (mScroller != null)
            mScroller.onSizeChanged(w, h, oldw, oldh);
    }


    public class IndexScroller extends DataSetObserver {

        private float mPreviewPadding;
        private float mDensity;
        private float mScaledDensity;
        private int mListViewWidth;
        private int mListViewHeight;
        private int mCurrentSection = -1, mMin = 26;
        private boolean mIsIndexing = false;
        private ListView mListView = null;
        private SectionIndexer mIndexer = null;
        private String[] mSections = null;
        private RectF mIndexbarRect;
        private float mSingle;
        private Context mContext;
        private Paint mIndexPaint;
        private float mHeight, mWidth, mGap, mBaseLineToTop, mMaxSingleWidth, mPadding;

        IndexScroller(Context context, ListView lv) {
            mContext = context;
            mDensity = context.getResources().getDisplayMetrics().density;
            mScaledDensity = context.getResources().getDisplayMetrics().scaledDensity;
            mListView = lv;
            setAdapter(mListView.getAdapter());
            mPreviewPadding = 1 * mDensity;
            mGap = 6 * mDensity;
            mPadding = 8 * mDensity;
            mIndexPaint = new Paint();
            mIndexPaint.setColor(ContextCompat.getColor(mContext, R.color.online_c66));
            mIndexPaint.setAntiAlias(true);
            mIndexPaint.setTextSize(TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP,
                    12f, mContext.getResources().getDisplayMetrics()));
            mIndexPaint.setTextAlign(Paint.Align.CENTER);
        }

        void draw(Canvas canvas) {
            if (mSections != null && mSections.length > 0) {
                // Preview is shown when mCurrentSection is set
                if (mCurrentSection >= 0) {
                    Paint previewPaint = new Paint();
                    previewPaint.setColor(Color.BLACK);
                    previewPaint.setAlpha(96);
                    previewPaint.setAntiAlias(true);
                    previewPaint.setShadowLayer(3, 0, 0, Color.argb(64, 0, 0, 0));

                    Paint previewTextPaint = new Paint();
                    previewTextPaint.setColor(Color.WHITE);
                    previewTextPaint.setAntiAlias(true);
                    previewTextPaint.setTextSize(50 * mScaledDensity);

                    float previewTextWidth = previewTextPaint.measureText(mSections[mCurrentSection]);
                    float previewSize = 2 * mPreviewPadding + previewTextPaint.descent() - previewTextPaint.ascent();
                    RectF previewRect = new RectF((mListViewWidth - previewSize) / 2
                            , (mListViewHeight - previewSize) / 2
                            , (mListViewWidth - previewSize) / 2 + previewSize
                            , (mListViewHeight - previewSize) / 2 + previewSize);

                    canvas.drawRoundRect(previewRect, 5 * mDensity, 5 * mDensity, previewPaint);
                    canvas.drawText(mSections[mCurrentSection], previewRect.left + (previewSize - previewTextWidth) / 2 - 1
                            , previewRect.top + mPreviewPadding - previewTextPaint.ascent() + 1, previewTextPaint);
                }
                for (int i = 0; i < mSections.length; i++) {
                    canvas.drawText(mSections[i], mIndexbarRect.left + mMaxSingleWidth / 2 + mPadding,
                            mIndexbarRect.top + i * (mSingle + mGap) + mBaseLineToTop, mIndexPaint);
                }
            }
        }

        boolean onTouchEvent(MotionEvent ev) {
            switch (ev.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    // If down event occurs inside index bar region, start indexing
                    if (contains(ev.getX(), ev.getY())) {

                        // It demonstrates that the motion event started from index bar
                        mIsIndexing = true;
                        // Determine which section the point is in, and move the list to that section
                        mCurrentSection = getSectionByPoint(ev.getY());
                        mListView.setSelection(mIndexer.getPositionForSection(mCurrentSection));
                        return true;
                    }
                    break;
                case MotionEvent.ACTION_MOVE:
                    if (mIsIndexing) {
                        // If this event moves inside index bar
                        if (contains(ev.getX(), ev.getY())) {
                            // Determine which section the point is in, and move the list to that section
                            mCurrentSection = getSectionByPoint(ev.getY());
                            mListView.setSelection(mIndexer.getPositionForSection(mCurrentSection));
                        }
                        return true;
                    }
                    break;
                case MotionEvent.ACTION_UP:
                    if (mIsIndexing) {
                        mIsIndexing = false;
                        mCurrentSection = -1;
                    }
                    break;
            }

            return false;
        }

        boolean contains(float x, float y) {
            // Determine if the point is in index bar region, which includes the right margin of the bar
            return (x >= mIndexbarRect.left && y >= mIndexbarRect.top && y <= mIndexbarRect.bottom);
        }

        void onSizeChanged(int w, int h, int oldw, int oldh) {
            mListViewWidth = w;
            mListViewHeight = h;
            final float margin = (mListViewHeight - mHeight) / 2;
            mIndexbarRect = new RectF(w - mWidth
                    , margin
                    , w
                    , h - margin);
        }

        private int getSectionByPoint(float y) {
            if (mSections == null || mSections.length == 0)
                return 0;
            return (int) Math.floor((y - mIndexbarRect.top) / (mSingle + mGap));
        }

        void setAdapter(Adapter adapter) {
            if (adapter instanceof SectionIndexer) {
                mIndexer = (SectionIndexer) adapter;
                mSections = (String[]) mIndexer.getSections();
                adapter.registerDataSetObserver(this);
                final Paint.FontMetrics fm = mIndexPaint.getFontMetrics();
                mSingle = fm.bottom - fm.top + fm.leading;
                mBaseLineToTop = mSingle - fm.bottom;
                mHeight = mSingle * mSections.length + mGap * (mSections.length - 1);
                for (String s : mSections) {
                    final float w = mIndexPaint.measureText(s);
                    if (mMaxSingleWidth < w) {
                        mMaxSingleWidth = w;
                    }
                }
                mWidth = mMaxSingleWidth + mPadding * 2;
            }
        }

        @Override
        public void onChanged() {
            mSections = (String[]) mIndexer.getSections();
            mListView.invalidate();
        }
    }
}
