package ezy.boost.update.download;

import android.Manifest;
import android.app.DownloadManager;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.Uri;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.widget.Toast;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

public class Updater {

    private File apkFile;
    private String title;
    private String downloadUrl;
    private Context context;
    private DownloadManager downloadManager;
    private long mTaskId;
    private boolean hideNotification = false;
    //    private ProgressListener mProgressListener;
    private boolean allowedOverRoaming = false;
    private DownloadReceiver downloadReceiver;
    private DownloadObserver downloadObserver;
    private boolean claerCache = false;

    private String[] perms = {Manifest.permission.WRITE_EXTERNAL_STORAGE,
            Manifest.permission.READ_EXTERNAL_STORAGE};
    private static final int RC_SDCARD = 123;

    private DownloadFailedReceiver downloadFailedReceiver = new DownloadFailedReceiver();

    private Updater(Context context) {
        this.context = context;
    }

    private void download() {
        if (context == null) {
            throw new NullPointerException("context must not be null");
        }

        if (TextUtils.isEmpty(downloadUrl)) {
            throw new NullPointerException("downloadUrl must not be null");
        }

        if (apkFile == null) {
            try {
                throw new Exception("文件下载地址为null");
            } catch (Exception e) {
                e.printStackTrace();
            }
            return;
        }

        if (downloadManager == null) {
            downloadManager = (DownloadManager) context.getSystemService(Context.DOWNLOAD_SERVICE);
        }

        DownloadManager.Request request = new DownloadManager.Request(Uri.parse(downloadUrl));

        request.setAllowedNetworkTypes(DownloadManager.Request.NETWORK_MOBILE | DownloadManager
                .Request.NETWORK_WIFI);

        request.setAllowedOverRoaming(allowedOverRoaming);

        request.setTitle(TextUtils.isEmpty(title) ? apkFile.getName() : title);

        request.setNotificationVisibility(hideNotification ? DownloadManager.Request.VISIBILITY_HIDDEN
                : DownloadManager.Request.VISIBILITY_VISIBLE);

        //设置下载路径
        request.setDestinationInExternalPublicDir(Environment.DIRECTORY_DOWNLOADS, apkFile.getName());

        //将下载请求加入下载队列
        //加入下载队列后会给该任务返回一个long型的id，
        //通过该id可以取消任务，重启任务等等
        mTaskId = downloadManager.enqueue(request);
        if (downloadFailedReceiver != null) {
            context.registerReceiver(downloadFailedReceiver,
                    new IntentFilter(Updater.DownloadFailedReceiver.tag));
        }
    }

    /**
     * 注册下载完成的监听
     */
    public void registerDownloadReceiver() {
        if (downloadReceiver == null) {
            downloadReceiver = new DownloadReceiver();
        }
        context.registerReceiver(downloadReceiver,
                new IntentFilter(DownloadManager.ACTION_DOWNLOAD_COMPLETE));

    }

    /**
     * 解绑下载完成的监听
     */
    public void unRegisterDownloadReceiver() {
        if (downloadReceiver != null) {
            context.unregisterReceiver(downloadReceiver);
        }
    }

    private ArrayList<ProgressListener> listeners;

    /**
     * 添加下载进度回调
     */
    public Updater addProgressListener(ProgressListener progressListener) {
        if (listeners == null) {
            listeners = new ArrayList<>();
        }
        if (!listeners.contains(progressListener)) {
            listeners.add(progressListener);
        }
        if (downloadObserver == null && handler != null && downloadManager != null) {
            downloadObserver = new DownloadObserver(handler, downloadManager, mTaskId);
            context.getContentResolver().registerContentObserver(Uri.parse("content://downloads/"),
                    true, downloadObserver);
        }
        return this;
    }

    CompleteListener completeListener;

    /***
     * 下载完成回调
     */
    public Updater bindCompleteListener(CompleteListener completeListener) {
        downloadReceiver.bindCompleteListener(completeListener);
        return this;
    }

    /**
     * 移除下载进度回调
     */
    public void removeProgressListener(ProgressListener progressListener) {
        if (!listeners.contains(progressListener)) {
            throw new NullPointerException("this progressListener not attch Updater");
        }
        if (listeners != null && !listeners.isEmpty()) {
            listeners.remove(progressListener);
            if (listeners.isEmpty() && downloadObserver != null)
                context.getContentResolver().unregisterContentObserver(downloadObserver);
        }
    }


    public static void showToast(Context context, String msg) {
        Toast.makeText(context, msg, Toast.LENGTH_SHORT).show();
    }

    private Handler handler = new Handler(new Handler.Callback() {
        @Override
        public boolean handleMessage(Message msg) {
            Bundle data = msg.getData();
            long cutBytes = data.getLong(DownloadObserver.CURBYTES);
            long totalBytes = data.getLong(DownloadObserver.TOTALBYTES);
            int progress = data.getInt(DownloadObserver.PROGRESS);
            if (listeners != null && !listeners.isEmpty()) {
                for (ProgressListener listener : listeners) {
                    listener.onProgressChange(totalBytes, cutBytes, progress);
                }
            }
            return false;
        }
    });

    public static class Builder {

        private Updater mUpdater;

        public Builder(Context context) {
            synchronized (Updater.class) {
                if (mUpdater == null) {
                    synchronized (Updater.class) {
                        mUpdater = new Updater(context);
                    }
                }
            }
        }

        /**
         * 设置apk下载的路径
         *
         * @param apkFile 自定义的全路径
         * @return
         */
        public Builder setApkFile(File apkFile) {
            if (apkFile == null)
                return this;
            mUpdater.apkFile = apkFile;
            return this;
        }

        /**
         * 设置下载的链接地址
         *
         * @param downloadUrl apk的下载链接
         * @return
         */
        public Builder setDownloadUrl(String downloadUrl) {
            mUpdater.downloadUrl = downloadUrl;
            return this;
        }

        /**
         * 通知栏显示的标题
         *
         * @param title 标题
         * @return
         */
        public Builder setNotificationTitle(String title) {
            mUpdater.title = title;
            return this;
        }

        /**
         * 隐藏通知栏
         *
         * @return
         */
        public Builder hideNotification() {
            mUpdater.hideNotification = true;
            return this;
        }

        /**
         * 是否为debug模式，会输出很多log信息（手动斜眼）
         *
         * @return
         */
        public Builder debug() {
            LogUtils.isDebug = true;
            return this;
        }

        /**
         * 允许漫游网络可下载
         *
         * @return
         */
        public Builder allowedOverRoaming() {
            mUpdater.allowedOverRoaming = true;
            return this;
        }


        public Builder clearCache() {
            mUpdater.claerCache = true;
            return this;
        }

        /**
         * 开始下载
         *
         * @return
         */
        public Updater start() {
            mUpdater.registerDownloadReceiver();
            mUpdater.download();
            return mUpdater;
        }

    }


    public class DownloadFailedReceiver extends BroadcastReceiver {

        public static final String tag = "DownloadFailedReceiver";

        @Override
        public void onReceive(Context context, Intent intent) {
            LogUtils.debug("开始重新下载");
            download();
        }
    }

}
